import vizact
import vizshape
import viz
from mymath import placeShapeInRing
import random as rd

class FlickerArray(object):
    def __init__(self, dpars, parent=None):
        # Generate the self.shapes:
        self.parent = parent
        self.position = dpars["POSITION"]
        self.group = viz.addGroup()
        if self.parent is None:
            self.group.setPosition(self.position)
        else: # FIXME:
            self.group.setParent(self.parent)
            self.group.setPosition([0,0,-5])
        self.n = dpars["N"] 
        self.resolution = dpars["RESOLUTION"] 
        self.stimulus_radius = dpars["STIMULUS_RADIUS"] 
        self.flicker_colors = dpars["FLICKER_COLORS"] 
        self._flicker_rate = dpars["FLICKER_RATE"] # note that this var will be controlled by the user: there should be an upper and downer limit
        self.range_flicker_rate = dpars["RANGE_FLICKER_RATE"]
        self.flicker_duration = dpars["FLICKER_DURATION"] # not used (that is manuall)
        self.nb_flickering = dpars["NB_FLICKERING"] 
        self.inner_circle = dpars["INNER_CIRCLE"] 
        self.outer_circle = dpars["OUTER_CIRCLE"] 
        self.rotation_speed = dpars["ROTATION_SPEED"]
        self.shapes = []
        self.flickering_shapes = []
        self.flickering_id = []

        for i in xrange(self.n):
            shape = vizshape.addCylinder(radius=self.stimulus_radius*20, axis=vizshape.AXIS_Z)
            shape.setParent(self.group)
            shape.setScale([1, 1, 0.001])
            shape.color([rd.uniform(0, 1) for j in range(3)])
            # a stimulus radius of zero will allow overlap:
            x, y = placeShapeInRing(self.shapes, self.inner_circle, self.outer_circle, self.stimulus_radius)
            # we used the increment the z position to avoid z-overlap glitches
            # that is similar to a z-order /layering system
            shape.setPosition([x, y, i ])
            self.shapes.append(shape)
            
            
        # select the self.shapes that will flicker:
        self.flickering_id = rd.sample(range(self.n), self.nb_flickering)
        self.flickering_shapes = [self.shapes[i] for i in self.flickering_id]

    def startFlickerAction(self, pool_index=0):
        # Created a sequence of color change actions
        # Note that pools are paralel threads:
        # we can add the actions to pool 0, 
        # so that we can add move actions on pool 1   
        if self.flicker_rate == 0:
            return        
        for shape in self.flickering_shapes:
            flicker_colors = rd.sample(self.flicker_colors, len(self.flicker_colors))
            color_sequence = [vizact.fadeTo(color, speed=float(self.flicker_rate)) for color in flicker_colors]
            flicker = vizact.sequence(color_sequence , viz.FOREVER)
            random_start = vizact.waittime(vizact.randfloat(0, len(self.flicker_colors)/self.flicker_rate))
            shape.addAction(random_start, pool_index)
            shape.addAction(flicker, pool_index) 

    def startRotationAction(self, pool_index=1):
        rotation1 = vizact.spin(0, 0, 1, self.rotation_speed, 2)
        rotation2 = vizact.spin(0, 0, 1, -self.rotation_speed, 2)
        rotation_sequence = vizact.sequence([rotation1, rotation2] , viz.FOREVER)
        for shape in self.shapes:
            shape.setCenter([-p for p in shape.getPosition()])
            shape.addAction(rotation_sequence, pool_index)

    def stopAction(self, pool_index, shape_index=[]):
        if shape_index == []:
            shape_index = xrange(self.n)
        for i in shape_index:
            self.shapes[i].endAction(pool_index)

    def show(self, visible, shape_index = []):
        if shape_index == []:
            shape_index = xrange(self.n)
        state = viz.ON if visible else viz.OFF
        for i in shape_index:
            self.shapes[i].visible(state)

    def changeFlickerGroup(self):
        # end the flickering of the flickerring objects
        for shape in self.shapes:
            shape.endAction(0) # flickering should be on the pool_index 0
        # select the next objects to flicker:
        if len(self.flickering_id) < (self.n/2):
            # if possible avoid to flicker the same objects
            to_sample = list(set(range(self.n)) - set(self.flickering_id))
        else:
            to_sample = range(self.n)
        self.flickering_shapes = [self.shapes[i] for i in rd.sample(to_sample, self.nb_flickering)]
        # make these objects flicker:
        self.startFlickerAction()

    @property
    def flicker_rate(self):
        return self._flicker_rate

    @flicker_rate.setter
    def flicker_rate(self, value):
        self._flicker_rate = max(self.range_flicker_rate[0], min(value, self.range_flicker_rate[1]))

    def renderOnlyToRenderNodes(self, list_camera, excludeMainPass=True):
        for s in self.shapes:
            s.renderOnlyToRenderNodes(list_camera, excludeMainPass)

    

